import json
import time
import torch
import logging
import argparse
import platform

from tqdm import tqdm
from pathlib import Path

from evaluators.chatgpt import ChatGPTEvaluator
from evaluators.geminipro import GeminiProEvaluator


def parse_arguments():
    parser = argparse.ArgumentParser()
    parser.add_argument('--model_name', default='geminipro',
                        choices=[
                            'gpt4',
                            'gpt3.5',
                            'geminipro',
                        ], type=str)
    parser.add_argument('--model_path', default=None, type=str)
    parser.add_argument('--cache_dir', default=None, type=str)
    parser.add_argument('--api_key', default='AIzaSyCKgHPsJ6kyJa0BBsY2QTBWZVBJcZRi5fY',
                        choices=[
                            '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',  # ChatGPT生产
                            'AIzaSyCKgHPsJ6kyJa0BBsY2QTBWZVBJcZRi5fY',  # GeminiPro
                        ], type=str)
    parser.add_argument('--inference_load_name', default='result_zh_gpt4.json', type=str)
    parser.add_argument('--judge_save_name', default=None, type=str)
    parser.add_argument('--log_save_name', default=None, type=str)
    args = parser.parse_args()

    return args


def judge_clinical_diagnosis(example):
    clinical_case_uid = example['clinical_case_uid']
    language = example['language']
    clinical_department = example['clinical_department']
    clinical_case_summary = example['clinical_case_summary']

    preliminary_diagnosis = example['preliminary_diagnosis']
    diagnostic_basis = example['diagnostic_basis']
    differential_diagnosis = example['differential_diagnosis']
    principal_diagnosis = example['principal_diagnosis']
    therapeutic_principle = example['therapeutic_principle']
    treatment_plan = example['treatment_plan']

    predicted_preliminary_diagnosis = example['predicted_preliminary_diagnosis']
    predicted_diagnostic_basis = example['predicted_diagnostic_basis']
    predicted_differential_diagnosis = example['predicted_differential_diagnosis']
    predicted_principal_diagnosis = example['predicted_principal_diagnosis']
    predicted_therapeutic_principle = example['predicted_therapeutic_principle']
    predicted_treatment_plan = example['predicted_treatment_plan']

    logging.info('===> [clinical diagnosis judge]...')
    logging.info('### [clinical case uid]: ' + str(clinical_case_uid))

    if language == 'zh':
        question = f'''您是一位经验丰富的{clinical_department}临床医生，根据给定的病例摘要，请分析并提供一个专业、详细、全面的临床诊断，包含以下6个部分：
1. 初步诊断：分条列出患者可能存在的一种或多种疾病的名称。
2. 诊断依据：分条列出您做出初步诊断的依据。
3. 鉴别诊断：分条列出几种可能导致患者目前症状的疾病，并简要说明为何将它们排除。如果您认为不需要进行鉴别诊断，请直接回答“诊断明确，无需鉴别。”。
4. 主要诊断：一种对患者身体健康危害最大、最需要治疗的疾病的名称。
5. 治疗原则：概述治疗您给出的主要诊断的一般原则。
6. 治疗计划：分条列出初步的治疗计划。

请确保您的回答符合以下格式：
“初步诊断：<您提供的初步诊断>
诊断依据：<您提供的诊断依据>
鉴别诊断：<您提供的鉴别诊断>
主要诊断：<您提供的主要诊断>
治疗原则：<您提供的治疗原则>
治疗计划：<您提供的治疗计划>”

以下是给定的病例摘要：

{clinical_case_summary}

您提供的临床诊断：'''
        solution = f'''初步诊断：{preliminary_diagnosis}
诊断依据：{diagnostic_basis}
鉴别诊断：{differential_diagnosis}
主要诊断：{principal_diagnosis}
治疗原则：{therapeutic_principle}
治疗计划：{treatment_plan}'''
        answer = f'''初步诊断：{predicted_preliminary_diagnosis}
诊断依据：{predicted_diagnostic_basis}
鉴别诊断：{predicted_differential_diagnosis}
主要诊断：{predicted_principal_diagnosis}
治疗原则：{predicted_therapeutic_principle}
治疗计划：{predicted_treatment_plan}'''
        prompt = f'''您是一位AI评估员，专门评估其他语言模型提供的回答的质量。您的主要目标是根据回答的流畅性、相关性、完整性和医学知识专业性来对回答进行评分。请使用以下评分标准来评估每个指标：

流畅性：
1分：完全破碎且无法阅读的句子片段
2分：大部分破碎，只有少量可读的词
3分：有一定流利度，但词汇有限
4分：在表达复杂主题方面基本上是流畅的
5分：人类水平流利度

相关性：
1分：与问题完全无关
2分：与问题有一定关系，但主要是离题的
3分：相关，但缺乏重点或关键细节
4分：高度相关，解决了大部分问题
5分：直接相关，准确地针对了问题

完整性：
1分：极度不完整
2分：几乎不完整，信息有限
3分：有一定的完整性，有一些信息
4分：大部分信息都已显示
5分：所有信息都已呈现

医学知识专业性：
1分：使用简单明了的语言，没有医学术语
2分：具有一些医学知识，但缺乏深入细节
3分：清晰地传达了一定的复杂医学信息
4分：对医学术语有扎实的认识，但有些细节错误
5分：在所有呈现的医学知识上都是完全正确的

您将会得到以下信息：
- 一个问题
- 这个问题的参考答案
- 一个模型对这个问题给出的回答

[问题]
{question}
[问题结束]

[参考答案]
{solution}
[参考答案结束]

[模型回答]
{answer}
[模型回答结束]

请确保以JSON格式提供您的的评估结果，并且仅提供JSON，分别对每个提到的指标进行评分，如下例所示：
{{"流畅性": 3, "相关性": 3, "完整性": 3, "专业性": 3}}'''
    else:
        question = f'''You are an experienced {clinical_department} clinician. Based on the given clinical case summary, please analyze and provide a professional, detailed, and comprehensive clinical diagnosis, including the following 6 parts:
1. Preliminary Diagnosis: List the names of one or multiple diseases that the patient might have.
2. Diagnostic Basis: List the basis for your preliminary diagnosis.
3. Differential Diagnosis: List several diseases that could cause the patient's current symptoms and briefly explain why you exclude them. If you believe differential diagnosis is unnecessary, please directly response "The diagnosis is clear and no differentiation is needed."
4. Principal Diagnosis: The name of a disease that is most harmful to the patient's physical health and needs immediate treatment.
5. Therapeutic Principle: Outline the general principles for treating your principal diagnosis.
6. Treatment Plan: List the rough treatment plan.

Please ensure that your response follows this format:
"Preliminary Diagnosis: <Your Preliminary Diagnosis>
Diagnostic Basis: <Your Diagnostic Basis>
Differential Diagnosis: <Your Differential Diagnosis>
Principal Diagnosis: <Your Principal Diagnosis>
Therapeutic Principle: <Your Therapeutic Principle>
Treatment Plan: <Your Treatment Plan>"

The following is the given clinical case summary:
{clinical_case_summary}

Your clinical diagnosis:'''
        solution = f'''Preliminary Diagnosis: {preliminary_diagnosis}
Diagnostic Basis: {diagnostic_basis}
Differential Diagnosis: {differential_diagnosis}
Principal Diagnosis: {principal_diagnosis}
Therapeutic Principle: {therapeutic_principle}
Treatment Plan: {treatment_plan}'''
        answer = f'''Preliminary Diagnosis: {predicted_preliminary_diagnosis}
Diagnostic Basis: {predicted_diagnostic_basis}
Differential Diagnosis: {predicted_differential_diagnosis}
Principal Diagnosis: {predicted_principal_diagnosis}
Therapeutic Principle: {predicted_therapeutic_principle}
Treatment Plan: {predicted_treatment_plan}'''
        prompt = f'''You are an AI evaluator specializing in assessing the quality of answers provided by other language models. Your primary goal is to rate the answers based on their fluency, relevance, completeness, proficiency in medicine. Use the following scales to evaluate each criterion:

Fluency:
1: Completely broken and unreadable sentence pieces
2: Mostly broken with few readable tokens
3: Moderately fluent but with limited vocabulary
4: Mostly coherent in expressing complex subjects
5: Human-level fluency

Relevance:
1: Completely unrelated to the question
2: Some relation to the question, but mostly off-topic
3: Relevant, but lacking focus or key details
4: Highly relevant, addressing the main aspects of the question
5: Directly relevant and precisely targeted to the question

Completeness:
1: Extremely incomplete
2: Almost incomplete with limited information
3: Moderate completeness with some information
4: Mostly complete with most of the information displayed
5: Fully complete with all information presented

Proficiency in medicine:
1: Using plain languages with no medical terminology
2: Equipped with some medical knowledge but lacking in-depth details
3: Conveying moderately complex medical information with clarity
4: Showing solid grasp of medical terminology but having some minor mistakes in detail
5: Fully correct in all presented medical knowledge

You will be provided with the following information:
- a question
- the solution to the question
- a model's answer to the question

[question]
{question}
[end of question]

[solution]
{solution}
[end of solution]

[answer]
{answer}
[end of answer]

Make sure to provide your evaluation results in JSON format and ONLY the JSON, with separate ratings for each of the mentioned criteria as in the following example:
{{"fluency": 3, "relevance": 3, "completeness": 3, "proficiency": 3}}'''

    prompt_tokens = evaluator.count_tokens(prompt)

    logging.info('### [prompt]: ' + str(prompt))
    logging.info('### [prompt tokens]: ' + str(prompt_tokens))

    try:
        response = evaluator.generate_text(prompt)

        logging.info('### [response]: ' + str(response))

        if response is None:
            response = ''
            response_tokens = 0
        else:
            response_tokens = evaluator.count_tokens(response)
        total_tokens = prompt_tokens + response_tokens

        logging.info('### [response tokens]: ' + str(response_tokens))
        logging.info('### [total tokens]: ' + str(total_tokens))

    except Exception as e:
        response = ''

        logging.error('### [Failed to generate text]: ' + e.__str__())

    example['prompt_judge_clinical_diagnosis'] = str(evaluator.format_prompt(prompt))
    example['judge_clinical_diagnosis'] = response

    return example


def judge_imaging_diagnosis(example):
    clinical_case_uid = example['clinical_case_uid']
    language = example['language']
    imageological_examination = example['imageological_examination']

    logging.info('===> [imaging diagnosis inference]...')
    logging.info('### [clinical case uid]: ' + str(clinical_case_uid))

    if isinstance(imageological_examination, dict):
        for imageological_examination_part_feature in imageological_examination.keys():
            findings = imageological_examination[imageological_examination_part_feature]['findings']
            impression = imageological_examination[imageological_examination_part_feature]['impression']
            predicted_impression = imageological_examination[imageological_examination_part_feature][
                'predicted_impression']
            if language == 'zh':
                imageological_examination_part_name = imageological_examination_part_feature_to_name_zh_dict[
                    imageological_examination_part_feature]
                question = f'''您是一位经验丰富的放射科医生，根据给定的{imageological_examination_part_name}检查报告中的影像所见部分，请分析并分条列出专业的影像诊断。
请确保您的回答高度简洁。

以下是给定的影像所见：
{findings}

您提供的影像诊断：'''
                solution = impression
                answer = predicted_impression
                prompt = f'''您是一位AI评估员，专门评估其他语言模型提供的回答的质量。您的主要目标是根据回答的流畅性、相关性、完整性和医学知识专业性来对回答进行评分。请使用以下评分标准来评估每个指标：

流畅性：
1分：完全破碎且无法阅读的句子片段
2分：大部分破碎，只有少量可读的词
3分：有一定流利度，但词汇有限
4分：在表达复杂主题方面基本上是流畅的
5分：人类水平流利度

相关性：
1分：与问题完全无关
2分：与问题有一定关系，但主要是离题的
3分：相关，但缺乏重点或关键细节
4分：高度相关，解决了大部分问题
5分：直接相关，准确地针对了问题

完整性：
1分：极度不完整
2分：几乎不完整，信息有限
3分：有一定的完整性，有一些信息
4分：大部分信息都已显示
5分：所有信息都已呈现

医学知识专业性：
1分：使用简单明了的语言，没有医学术语
2分：具有一些医学知识，但缺乏深入细节
3分：清晰地传达了一定的复杂医学信息
4分：对医学术语有扎实的认识，但有些细节错误
5分：在所有呈现的医学知识上都是完全正确的

您将会得到以下信息：
- 一个问题
- 这个问题的参考答案
- 一个模型对这个问题给出的回答

[问题]
{question}
[问题结束]

[参考答案]
{solution}
[参考答案结束]

[模型回答]
{answer}
[模型回答结束]

请确保以JSON格式提供您的的评估结果，并且仅提供JSON，分别对每个提到的指标进行评分，如下例所示：
{{"流畅性": 3, "相关性": 3, "完整性": 3, "专业性": 3}}'''
            else:
                imageological_examination_part_name = imageological_examination_part_feature_to_name_en_dict[
                    imageological_examination_part_feature]
                question = f'''You are an experienced radiologist. Based on the findings section of the given {imageological_examination_part_name} examination report, please analyze and list professional impression.
Please ensure that your response is highly concise.

The following is the given findings:
{findings}

Your impression:'''
                solution = impression
                answer = predicted_impression
                prompt = f'''You are an AI evaluator specializing in assessing the quality of answers provided by other language models. Your primary goal is to rate the answers based on their fluency, relevance, completeness, proficiency in medicine. Use the following scales to evaluate each criterion:

Fluency:
1: Completely broken and unreadable sentence pieces
2: Mostly broken with few readable tokens
3: Moderately fluent but with limited vocabulary
4: Mostly coherent in expressing complex subjects
5: Human-level fluency

Relevance:
1: Completely unrelated to the question
2: Some relation to the question, but mostly off-topic
3: Relevant, but lacking focus or key details
4: Highly relevant, addressing the main aspects of the question
5: Directly relevant and precisely targeted to the question

Completeness:
1: Extremely incomplete
2: Almost incomplete with limited information
3: Moderate completeness with some information
4: Mostly complete with most of the information displayed
5: Fully complete with all information presented

Proficiency in medicine:
1: Using plain languages with no medical terminology
2: Equipped with some medical knowledge but lacking in-depth details
3: Conveying moderately complex medical information with clarity
4: Showing solid grasp of medical terminology but having some minor mistakes in detail
5: Fully correct in all presented medical knowledge

You will be provided with the following information:
- a question
- the solution to the question
- a model's answer to the question

[question]
{question}
[end of question]

[solution]
{solution}
[end of solution]

[answer]
{answer}
[end of answer]

Make sure to provide your evaluation results in JSON format and ONLY the JSON, with separate ratings for each of the mentioned criteria as in the following example:
{{"fluency": 3, "relevance": 3, "completeness": 3, "proficiency": 3}}'''

            prompt_tokens = evaluator.count_tokens(prompt)

            logging.info('### [prompt]: ' + str(prompt))
            logging.info('### [prompt tokens]: ' + str(prompt_tokens))

            try:
                response = evaluator.generate_text(prompt)

                logging.info('### [response]: ' + str(response))

                if response is None:
                    response = ''
                    response_tokens = 0
                else:
                    response_tokens = evaluator.count_tokens(response)
                total_tokens = prompt_tokens + response_tokens

                logging.info('### [response tokens]: ' + str(response_tokens))
                logging.info('### [total tokens]: ' + str(total_tokens))

            except Exception as e:
                response = ''

                logging.error('### [Failed to generate text]: ' + e.__str__())

            imageological_examination[imageological_examination_part_feature]['prompt_judge_imaging_diagnosis'] = str(
                evaluator.format_prompt(prompt))
            imageological_examination[imageological_examination_part_feature]['judge_imaging_diagnosis'] = response

    return example


def main():
    with open(inference_load_path, mode='r', encoding='utf-8') as file:
        dataset = json.load(file)

    for index in tqdm(range(len(dataset))):
        if dataset[index]['clinical_case_uid'] in choose_clinical_case_uid_list:
            dataset[index] = judge_clinical_diagnosis(dataset[index])
            dataset[index] = judge_imaging_diagnosis(dataset[index])

            with open(judge_save_path, mode='w', encoding='utf-8') as file:
                json.dump(dataset, file, ensure_ascii=False, indent=2)


if __name__ == '__main__':
    args = parse_arguments()

    timestamp = time.strftime('%Y-%m-%d-%H-%M-%S')
    inference_dir = Path(__file__).parent.parent / Path('inferences')
    if not inference_dir.is_dir():
        inference_dir.mkdir(parents=True, exist_ok=True)
    inference_load_path = inference_dir / Path(args.inference_load_name)
    judge_dir = Path(__file__).parent.parent / Path('judges')
    if not judge_dir.is_dir():
        judge_dir.mkdir(parents=True, exist_ok=True)
    if args.judge_save_name is None:
        judge_save_path = judge_dir / Path(
            'judge_' + args.inference_load_name.split('.')[0] + f'_by_{args.model_name}({timestamp}).json')
    else:
        judge_save_path = judge_dir / Path(
            'judge_' + args.judge_save_name.split('.')[0] + f'_by_{args.model_name}({timestamp}).json')
    log_dir = Path(__file__).parent.parent / Path('logs')
    if not log_dir.is_dir():
        log_dir.mkdir(parents=True, exist_ok=True)
    if args.log_save_name is None:
        log_save_path = log_dir / Path(
            'judge_' + args.inference_load_name.split('.')[0] + f'_by_{args.model_name}({timestamp}).log')
    else:
        log_save_path = log_dir / Path(
            'judge_' + args.judge_save_name.split('.')[0] + f'_by_{args.model_name}({timestamp}).log')

    logger = logging.getLogger()
    logger.setLevel(logging.INFO)
    formatter = logging.Formatter(fmt='%(asctime)s - %(filename)s - %(levelname)s - %(message)s',
                                  datefmt='%Y-%m-%d %H:%M:%S')
    file_handler = logging.FileHandler(filename=log_save_path, mode='w', encoding='utf-8')
    file_handler.setLevel(logging.INFO)
    file_handler.setFormatter(formatter)
    stream_handler = logging.StreamHandler()
    stream_handler.setLevel(logging.INFO)
    stream_handler.setFormatter(formatter)
    logger.addHandler(file_handler)
    logger.addHandler(stream_handler)

    device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')
    device_name = torch.cuda.get_device_name(
        torch.cuda.current_device()) if torch.cuda.is_available() else platform.processor()

    if args.model_name == 'gpt4':
        if args.model_path is None:
            pretrained_model_name_or_path = 'gpt-4-1106-preview'
        else:
            pretrained_model_name_or_path = args.model_path
        evaluator = ChatGPTEvaluator(pretrained_model_name_or_path=pretrained_model_name_or_path, api_key=args.api_key)
    elif args.model_name == 'gpt3.5':
        if args.model_path is None:
            pretrained_model_name_or_path = 'gpt-3.5-turbo-1106'
        else:
            pretrained_model_name_or_path = args.model_path
        evaluator = ChatGPTEvaluator(pretrained_model_name_or_path=pretrained_model_name_or_path, api_key=args.api_key)
    elif args.model_name == 'geminipro':
        if args.model_path is None:
            pretrained_model_name_or_path = 'models/gemini-pro'
        else:
            pretrained_model_name_or_path = args.model_path
        evaluator = GeminiProEvaluator(pretrained_model_name_or_path=pretrained_model_name_or_path,
                                       api_key=args.api_key)
    else:
        pretrained_model_name_or_path = None
        evaluator = None
        logger.error('### [model name not found]')
        exit()

    logging.info('### [device]: ' + str(device))
    logging.info('### [device name]: ' + str(device_name))
    logging.info('### [model name]: ' + str(args.model_name))
    logging.info('### [model path]: ' + str(pretrained_model_name_or_path))

    imageological_examination_part_name_zh_to_feature_dict = {
        'CT平扫': 'plain_computed_tomography_scan',
        'CT增强': 'contrast_computed_tomography_scan',
        'CT平扫+增强': 'plain_computed_tomography_scan+contrast_computed_tomography_scan',
        '彩色多普勒超声': 'color_doppler_ultrasound',
        '乳腺钼靶': 'mammography',
        '磁共振平扫': 'plain_magnetic_resonance_imaging_scan',
        '磁共振增强': 'contrast_magnetic_resonance_imaging_scan',
        '磁共振平扫+增强': 'plain_magnetic_resonance_imaging_scan+contrast_magnetic_resonance_imaging_scan',
        '磁共振血管造影': 'magnetic_resonance_angiography',
        '磁共振平扫+血管造影': 'plain_magnetic_resonance_imaging_scan+magnetic_resonance_angiography',
        '磁共振水成像': 'magnetic_resonance_hydrography',
        '磁共振平扫+水成像': 'plain_magnetic_resonance_imaging_scan+magnetic_resonance_hydrography',
        '数字X线摄影': 'digital_radiography',
        'CT血管造影': 'computed_tomography_angiography',
        '超声心动图': 'echocardiogram',
        '冠状动脉造影': 'coronary_arteriography',
        '食道、胃、十二指肠镜': 'esophagogastroduodenoscopy',
        '甲状旁腺显像': 'parathyroid_imaging',
        '电子鼻咽喉镜': 'nasopharyngoscope',
        '消化道造影': 'gastrointestinal_tract_radiography'
    }
    imageological_examination_part_feature_to_name_zh_dict = {value: key for key, value in
                                                              imageological_examination_part_name_zh_to_feature_dict.items()}

    imageological_examination_part_feature_to_name_en_dict = {
        'plain CT scan': 'plain_computed_tomography_scan',
        'contrast CT scan': 'contrast_computed_tomography_scan',
        'plain CT scan and contrast CT scan': 'plain_computed_tomography_scan+contrast_computed_tomography_scan',
        'color doppler ultrasound': 'color_doppler_ultrasound',
        'mammography': 'mammography',
        'plain MRI scan': 'plain_magnetic_resonance_imaging_scan',
        'contrast MRI scan': 'contrast_magnetic_resonance_imaging_scan',
        'plain MRI scan and contrast MRI scan': 'plain_magnetic_resonance_imaging_scan+contrast_magnetic_resonance_imaging_scan',
        'MRA': 'magnetic_resonance_angiography',
        'plain MRI scan and MRA': 'plain_magnetic_resonance_imaging_scan+magnetic_resonance_angiography',
        'MR hydrography': 'magnetic_resonance_hydrography',
        'plain MRI scan and MR hydrography': 'plain_magnetic_resonance_imaging_scan+magnetic_resonance_hydrography',
        'digital radiography': 'digital_radiography',
        'CT angiography': 'computed_tomography_angiography',
        'echocardiogram': 'echocardiogram',
        'coronary arteriography': 'coronary_arteriography',
        'esophagogastroduodenoscopy': 'esophagogastroduodenoscopy',
        'parathyroid imaging': 'parathyroid_imaging',
        'nasopharyngoscope': 'nasopharyngoscope',
        'gastrointestinal tract radiography': 'gastrointestinal_tract_radiography'
    }
    imageological_examination_part_feature_to_name_en_dict = {value: key for key, value in
                                                              imageological_examination_part_feature_to_name_en_dict.items()}

    clinical_department_zh_to_en_dict = {
        '乳腺外科': 'breast surgical department',
        '产科': 'obstetrics department',
        '儿科': 'pediatrics department',
        '内分泌内科': 'endocrinology department',
        '呼吸内科': 'respiratory medicine department',
        '妇科': 'gynecology department',
        '心脏外科': 'cardiac surgical department',
        '心血管内科': 'cardiovascular medicine department',
        '泌尿外科': 'urinary surgical department',
        '消化内科': 'gastroenterology department',
        '甲状腺外科': 'thyroid surgical department',
        '疝外科': 'hernia surgical department',
        '神经内科': 'neurology department',
        '神经外科': 'neurosurgery department',
        '耳鼻咽喉头颈外科': 'otolaryngology head and neck surgical department',
        '肛门结直肠外科': 'anus and intestine surgical department',
        '肝胆胰外科': 'hepatobiliary and pancreas surgical department',
        '肾内科': 'nephrology department',
        '胃肠外科': 'gastrointestinal surgical department',
        '胸外科': 'thoracic surgical department',
        '血液内科': 'hematology department',
        '血管外科': 'vascular surgical department',
        '骨科': 'orthopedics department',
    }
    clinical_department_zh_list = ''
    for _ in clinical_department_zh_to_en_dict.keys():
        clinical_department_zh_list += f'- {_}\n'
    clinical_department_en_list = ''
    for _ in clinical_department_zh_to_en_dict.values():
        clinical_department_en_list += f'- {_}\n'

    choose_clinical_case_uid_list = ['02f44d8903084e60a0f99dcda5bcf9db', '3faeed5576ee4af48e5cc24fe4a62287', '6c9bb0a4cdf14527992a33650644c474', 'd6b83de7be7f420bb236e9386a957b72',
                                     'df10ac2333fb46c58ecc9bef01ad5b35', 'e7209a8a35e24de388bf6cbb188659e0', 'a3fdb884378242299380a2e8e9b6ef67', '332a965d05134d2b8d98adfaa5bd1a4d',
                                     'df0da11d54b9415eba4ed3f7ef8f9a57', 'ad0bafc4d7f5460480e51db7aec9a420', 'b9cd720b36df4045908cdea51e109931', 'c1cb3becfd2644d28b4942ce861163d0',
                                     '141667f866d3424cb4e9f08f1efba65f', '17745a2aa5d84da1986559235703dff1', '799af1812d42424d9b47edad171d82ba', '886347fc423b4a3989a22d1df71c3fa1',
                                     'b68e11c0427b4223a75efb64ec5cedf9', 'a4deec95bbd14a5aae11ac4b647e2348', 'c49c15043f8546699ccc6c1cb1a6141e', '8e4380583c3a4cfd9f57755b67318da0',
                                     '5f29cccee56740f39569957148c0dc0d', '500aed0b82704977b0b3d4cfc2996d94', '6f5b146a331844bf8e6a2bb2cb081452', '42db4d30ed5b4ef1859e83f9dcdead82',
                                     '1b246bb198cd4d0bbce956342233ed55', 'd6a8347f03c54e66bea1ee2c1bf96eb7', '09341b4e236b45c6a2be7d1d1cbe0ac5', 'e7a082a25268417e869085c2fe87dfa6',
                                     '740342fb075d4c9cbd2af63e7eb17767', 'bf58697d094a4f148a6ce88c4d7510ec', '62034b6e268f4f189620f093c30619ae', 'ffaaab9fa35d45f59c04ff9c02ed076b',
                                     '31a483c38f4e4ba4b4c4924ed14cbae6', 'a7c642ca02ec434a972a1b39df9bc8d0', '276057f22da24aea9c0f658b66505a43', '23719caf760244b1b190d32e66a961c7',
                                     '8794002df418405e97f73877aba13b6d', '4a30e57e565e4863801dc6f74786fc37', 'ae1f91894d7b45ae92abdfee832b429a', '767504eb77fa456fadec295af717e7ef',
                                     'ad44be7fb7864ab89385ce033561325e', 'ec925f2f2ec5438d991869032a47bc7a', 'fefdc222d71e4e68876679b1178d113d', '117d57b4c1e64097a51dc29a94c8f0be',
                                     '23ef0751c9a64ad1921958dc00708ff2', '4afa8231b36a49d7a712b242f5849b24', '6936a98a81bf430484c297647a7c2e30', '20263fa0b7aa48978f90d5c46c674fa9',
                                     'c0b86381689b4f19892658ca18249881', '81a1abb6bfc7432a8b25047ca10d7ff7', '91dd236832764acb82ae5ff74ae01063', 'e33fd4605cec4771a750bce9eb92ff38',
                                     '3efd19eead9146699acd96c37bc563d1', '33ef80d49ed3491c889adab5a6f7ce6e', '761369fddc844032b4d0eb8c95acebee', '5e781a437f5846d6b273a9259d58e72e',
                                     '25a4f426a8fd4c418835765c7699a130', '94d8abab8a4643cc91c2443e96f00027', '1c946352b7014b40a81f815f7ff06a26', 'fe555cb9c33e4d93b2b4012516c69d0e',
                                     '0e887c2fdb714d9cb0c1a27d45d8b890', 'f7f3f9b5ad054f579e94e0d29c2a42a9', '2ef8fc2c1f5d40f8a9676c6f2d6d6f69', '3cde10f267914cefa497b1e6e939ab67',
                                     '4f5a67d1eb4e424892eacac92ead4285', '8074a8da21bd42eba90ec55c662e65df', '7cf3d52524ac4cbb970c86a6150b680d', '67ca0ff193e142c484444f3071f59e11',
                                     'fdb82ebb0c7d4723b9bd2c35bcb17a60', '302ea4f048df47e2a0a64b51aca6d70c', 'e913c1388f1546e88bdd5bd4560f634e', '291c3737bbb64555a5a7251b540e0186',
                                     '7706a04e4494472e8717cd539c355375', '306eb172fcd34e3d90a06b0e7776b8f4', 'ed05276db7fc410dbed5da3152e4c000', '5e4c0f0781ef49fca2a95c36ac9c05f3',
                                     '0cb22862a26c430eb1841d8e6b2ef66d', 'b3c9e046aa2b4ad3860858a750864a85', 'b707c7332dc74965b0fc93bb9d72c40d', 'f5ee4a04929f46caad71938389dd160a',
                                     'cc9d4fafb1a749008e6db23b7d186652', 'e79cc90ece864933a6a1486c6403c0fe', 'ded9247fef134b98a7b0d28877877ab2', 'a64dcbc4f86e4a7da25c795dfb3f1169',
                                     'c8a96a0e2d894b5380e03d2dba28c795', 'da31596b546249899b077a5ba91662e7', 'fbd447f58e264cfc84df8f8a596003bb', '24107305ba5b4274961d43bcc7b21abb',
                                     'fef6d537653b4b30872536879624fd90', '28affa5510484ac18b502af672587bd6', 'ba3b867e45164498b20b26bb08796c22', '21f7f573964a402e84abaca1440bb4a9',
                                     '8dba5a15092a460cba4f748d838e00ca', '6e02de8299614bd786e55a4412c152df', 'e75945fc8b80457286d4c3536592949d', '0ff0f7a139fa48ee82026cff4c3bf1e0',
                                     '45d22746bf094e0582bf47e5891ac49a', '82239e3738e64233a6f4e0ae272290eb', '1b31b766acdc4c479a6d0bed8e5141e6', 'c10559da9df94e69b855bd70daf0ddcc',
                                     'e4bacf01d77b4e9cb08b2c0422b3b3aa', '88b86ce6f5a34edc88524ce86652f9c5', '9c3c3c19515e4d0594d1a883338b44a5', 'b604d357937d45f29297692da3104641',
                                     '08640e0de9d649ebbb0c41764391fac5', '18701ef6020a43a399c43d58abd26d65', '5b53edbf5e964d87ad7f0801f4c264c2', '54c1bc8338874b0caa11bf215a01363c',
                                     '97db0b630a304414aa40575fb3f14686', 'd11fbb10780a4948b8716c3099b23505', '8e8816fcc7aa4b279e6e972be3305d73', '6a723a95b6e4465eb3cb031be96460cb',
                                     '53d0685f335f418389ac8321d2890877', '39f1d81666404949878c3c6062c332a5', '0e93d178bcec44cab8a47c70abac839a', '24d914d6a7c6489e914aede55ae4e9d8',
                                     '228dc109e1d4479aa3e4fa6844ed8e36', '0fad21ac8a6a4d79b4d93e24e0f7af9a', 'a7678541f00245b4ac4b02d0b46903e4', '0945713082af41248efbfdcbd6e12c2c',
                                     'e2c3c29e95954cbea35ab671cc5cc95d', 'dab9be7a8d654d31b02e41f97a047ce2', 'd463eb84751e47da8a2fe518b8ec8910', '194c1fa97a274d64a94b2160597d46a5',
                                     '8f30cbb011f140219b9259b010992691', 'f8688ecbc40c4dfe87cb58ff7d9cd211', '5a7834026a96437ebd2aff980d42900e', '5f68080ef9554deba86011c8b0be58af',
                                     '9e0012ca5e784342986ff409a7d9c5c8', 'fc70237d28e6411bb4f249a99ba4ab71', 'f25073ce455344c3aa4a2232c75e3748', '03a5b961b4e84662975daed75c518900',
                                     '31dc1f6562044ff1b5fdbacbb26e5b0e', 'd9f4d33d9ed045cba18e7304afb9e293', 'cb24a62608bf43edba92674acbd9713a', '981d46239fec4345aeafb08991f4be00',
                                     '8e480098e8384b42b8951eeec5a8813c', 'c75c2f308e5d40daba172956627a7447', 'ef32cb78e8b64378921d2d8c5d3700c4', '31a99abb8b0a4ab9ab05efd16541167f',
                                     '3ae0fea3de0641199a8ed81ced292451', 'bb1d7a8a86dd40e192c035e5d6bb2375', 'b0dcc8c3a8354e26ba6888ab30612a79', 'ea4dbcc68b794729857814079d2b4104',
                                     '44ccfbb14c9a432b9f62803aae1c35d1', '017ff34ad9b04639a65ba22379934f5e', 'c5e5c6595b3b491d9ac1ae93a2c6af32', 'c59476693e854ff3945fbfd37ce96767',
                                     '18e7eda6c8f4446c8507cd5b69ce6400', '1b01c7d72ac2421faf3324db09da7ba0']

    main()
